/****************************************************************************
 *   LPC8xx 4 channel PWM
 *
 *   Show simple four-channel PWM generation.
 *   Uses the unified 32-bit timer mode to generate single-edge aligned outputs.
 *   Channels can have different polarity. Demo state machine has been configured
 *   for positive pulses at CTOUT_0/1 and negative pulses at CTOUT_2/3.
 *   CTIN_0 is used a #ABORT input. If low, forces the outputs to their idle states,
 *   halts the timer, and generates an interrupt.
 *   An interrupt can be generated after each cycle.
****************************************************************************/

#include "board.h"

extern void sct_fsm_init (void);

#define SCT_IRQ_EVENT_IRQ_cycle (0)
#define SCT_IRQ_EVENT_IRQ_abort (1)
 
#define C_pwm_cycle (1000000)
#define C_pwm_val1 	(400000)
#define C_pwm_val2 	(500000)
#define C_pwm_val3 	(100000)
#define C_pwm_val4 	(900000)

volatile uint32_t numPwmCycles;
volatile int pwmAborted;
volatile int pwmPhase;

void sct_fsm_init (void)
{

	Chip_SCT_Init(LPC_SCT);														/* enable the SCT clock     */

	Chip_SCT_Config(LPC_SCT, SCT_CONFIG_AUTOLIMIT_L		|
                           SCT_CONFIG_32BIT_COUNTER );						/* AUTOLIMIT_L UNIFIED */

	Chip_SCT_SetMatchCount(LPC_SCT, SCT_MATCH_0, C_pwm_cycle);						/* match_0_ */
	Chip_SCT_SetMatchReload(LPC_SCT, SCT_MATCH_0, C_pwm_cycle);

	Chip_SCT_SetMatchCount(LPC_SCT, SCT_MATCH_1, C_pwm_val1);						/* match_0_ */
	Chip_SCT_SetMatchReload(LPC_SCT, SCT_MATCH_1, C_pwm_val1);

	Chip_SCT_SetMatchCount(LPC_SCT, SCT_MATCH_2, C_pwm_val2);						/* match_0_ */
	Chip_SCT_SetMatchReload(LPC_SCT, SCT_MATCH_2, C_pwm_val2);

	Chip_SCT_SetMatchCount(LPC_SCT, SCT_MATCH_3, C_pwm_val3);						/* match_0_ */
	Chip_SCT_SetMatchReload(LPC_SCT, SCT_MATCH_3, C_pwm_val3);

	Chip_SCT_SetMatchCount(LPC_SCT, SCT_MATCH_4, C_pwm_val4);						/* match_0_ */
	Chip_SCT_SetMatchReload(LPC_SCT, SCT_MATCH_4, C_pwm_val4);


	/* OUTPUT registers */

	Chip_SCT_SetOutput(LPC_SCT, SCT_OUTPUT_0, SCT_EVT_0);  			 /* Output_pin_0 */
	Chip_SCT_ClearOutput(LPC_SCT, SCT_OUTPUT_0, (CHIP_SCT_EVENT_T)  ( SCT_EVT_1 |
                                                                    SCT_EVT_5 ));

	Chip_SCT_SetOutput(LPC_SCT, SCT_OUTPUT_1, SCT_EVT_0);  			 /* Output_pin_0 */
	Chip_SCT_ClearOutput(LPC_SCT, SCT_OUTPUT_1, (CHIP_SCT_EVENT_T)  ( SCT_EVT_2 |
                                                                    SCT_EVT_5 ));

	Chip_SCT_SetOutput(LPC_SCT, SCT_OUTPUT_2, (CHIP_SCT_EVENT_T)  ( SCT_EVT_3	|
                                                                  SCT_EVT_5 ));  			 /* Output_pin_0 */
	Chip_SCT_ClearOutput(LPC_SCT, SCT_OUTPUT_2, SCT_EVT_0);

	Chip_SCT_SetOutput(LPC_SCT, SCT_OUTPUT_3, (CHIP_SCT_EVENT_T)  ( SCT_EVT_4	|
                                                                  SCT_EVT_5 ));  			 /* Output_pin_0 */
	Chip_SCT_ClearOutput(LPC_SCT, SCT_OUTPUT_3, SCT_EVT_0);


	Chip_SCT_Output(LPC_SCT, SCT_EVT_2 | SCT_EVT_3);

	/* EVENT registers */

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_0, 0xFFFFFFFF);			/* U: --> state state_1 */
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_0,	(CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH0	|
                                                                        SCT_COMBMODE_MATCH	));

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_1, 0xFFFFFFFF);			/* U: --> state state_1 */
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_1, (CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH1	|
                                                                        SCT_COMBMODE_MATCH	));

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_2, 0xFFFFFFFF);			/* U: --> state state_1 */
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_2, (CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH2	|
                                                                        SCT_COMBMODE_MATCH	));

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_3, 0xFFFFFFFF);			/* U: --> state state_1 */
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_3, (CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH3	|
                                                                        SCT_COMBMODE_MATCH	));

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_4, 0xFFFFFFFF);			/* U: --> state state_1 */
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_4, (CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH4	|
                                                                        SCT_COMBMODE_MATCH	));

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_5, ENABLE_ALL_STATES);			/* U: --> state state_1 */
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_5, (CHIP_SCT_EVENTCTRL_T)  ( SCT_IOCOND_LOW	|
                                                                        SCT_COMBMODE_IO	));

	LPC_SCT->HALT_L =  SCT_EVE_5;
	LPC_SCT->LIMIT_L = SCT_EVE_5;

	Chip_SCT_EnableEventInt(LPC_SCT, (CHIP_SCT_EVENT_T) ( SCT_EVT_0	| SCT_EVT_5 ));

	Chip_SCT_SetConflictResolution(LPC_SCT, 0 , 0x5A);

	NVIC_EnableIRQ(SCT_IRQn);

	Chip_SCT_ClearControl(LPC_SCT , SCT_CTRL_HALT_L);

}



void SCT_IRQHandler(void)
{
  uint32_t status = Chip_SCT_GetEventFlag(LPC_SCT);

    if (status & (1 << 0))
    {
        ++numPwmCycles;                                    	// interrupt once per PWM cycle
    }
    if (status & (1u << 5))
    {
        pwmAborted = 1;                                    	// Abort interrupt
    }
    LPC_SCT->EVFLAG = status;                             // clear interrupt flag
}



int main (void)
{
	uint32_t lastCycles;

	SystemCoreClockUpdate();
	Board_Init();



	Chip_Clock_EnablePeriphClock(SYSCTL_CLOCK_IOCON);										// enable clock for IOCON
	Chip_SWM_Init();

  Chip_SWM_DisableFixedPin(SWM_FIXED_ACMP_I1);                      //disable Pinmux for P0_0 (ACMP_I1)
	Chip_SWM_MovablePinAssign(SWM_CTIN_0_I, 0);												// SCT_IN0  at P0_0 used as ABORT
	Chip_SWM_MovablePinAssign(SWM_CTOUT_0_O, 17);											// SCT_OUT0 at P0_17  = green trace  = PWM1
	Chip_SWM_MovablePinAssign(SWM_CTOUT_1_O, 7);											// SCT_OUT1 at P0_7  = red trace    = PWM2
	Chip_SWM_MovablePinAssign(SWM_CTOUT_2_O, 16);											// SCT_OUT2 at P0_16  = yellow trace = PWM3
	Chip_SWM_MovablePinAssign(SWM_CTOUT_3_O, 12);											// SCT_OUT3 at P0_12 = blue trace   = PWM4

	Chip_IOCON_PinEnableHysteresis(LPC_IOCON, IOCON_PIO0);
	

	sct_fsm_init();                                   										// Initialize SCT


    while (1)                                              									// loop forever
    {
		if (numPwmCycles != lastCycles)
		  {
			lastCycles = numPwmCycles;
			if ((lastCycles % 5) == 0)                 										// every few PWM cycles change the duty cycles
			{
				Chip_SCT_SetClrControl(LPC_SCT, SCT_CONFIG_NORELOADL_U, ENABLE);			// NORELOAD_L (U)
			   if (pwmPhase == 0)
			   {
				  Chip_SCT_SetMatchReload(LPC_SCT, SCT_MATCH_1,  200000);
				  Chip_SCT_SetMatchReload(LPC_SCT, SCT_MATCH_3,  700000);
			   }
			   else
			   {
				  Chip_SCT_SetMatchReload(LPC_SCT, SCT_MATCH_1,  950000);
				  LPC_SCT->MATCHREL[3].U = Chip_SCT_GetMatchReload(LPC_SCT, SCT_MATCH_0);	// duty cycle 0 (test conflict resolution)
			   }
			   	   Chip_SCT_SetClrControl(LPC_SCT, SCT_CONFIG_NORELOADL_U, DISABLE);		// NORELOAD_L (U)
			   	   ++pwmPhase;
			   if (pwmPhase > 1)
			   {
				   pwmPhase = 0;
			   }
			}
		   }
		   if (pwmAborted)          /* Demo: Poll ABORT input, and restart timer if abort condition has gone. */
		   {
			   while (!(LPC_SCT->INPUT & (1 << 0))) ;

			   Chip_SCT_ClearControl(LPC_SCT,SCT_CTRL_HALT_L);		// start timer
			   pwmAborted = 0;
		   }

	//	   __WFI();
	 }
}
